
#include "p33Fxxxx.h"
#include "sram.h"
#include "dac.h"
#include "adc.h"
#include "infrared.h"
#include "config.h"
#include "DEE Emulation 16-bit.h"
#include <string.h>
#include <stdlib.h>

/*
  SportSync firmware, Silicon Chip 2011
  Chip: dsPIC33FJ64GP802

  Pin configuration:
  
  1:  MCLR     +3.3V
  2:  RA0/AN0  Audio input (analog)	
  3:  RA1      CS/RCK for 2x74HC595 memory address latches
  4:  RB0/RP0  SDO for 2x74HC595 memory address latches & control for "buffering" LED during idle (active high)
  5:  RB1/RP1  SCK for 2x74HC595 memory address latches & control for "OK" LED during idle (active high)
  6:  RB2      Infrared data input
  7:  RB3      Memory data bus
  8:  VSS      0V
  9:  RA2/CLKI 8MHz clock signal
  10: RA3      Memory OE-bar (also brings CE-bar low if brought low)
  11: RB4      Memory data bus
  12: RA4      Memory WR-bar (also brings CE-bar low if brought low)
  13: VDD      +3.3V
  14: RB5      Memory data bus
  15: RB6      Memory data bus
  16: RB7      Memory data bus
  17: RB8      Memory data bus
  18: RB9      Memory data bus
  19: VSS      0V
  20: VCAP     10uF tantalum capacitor to VSS
  21: RB10     Memory data bus
  22: RB11     Memory address bus
  23: RB12     Memory address bus
  24: RB13     Memory address bus
  25: DAC1LP   Analog output +
  26: DAC1LN   Analog output -
  27: AVSS     0V
  28: AVDD     +3.3V (RC filtered)
*/

/*
  Clock = 8MHz
  PLLPRE = 0  (N1 = 2)
  PLLPOST = 0 (N2 = 2)
  PLLDIV = 38 (M = 40)
  Fosc = 40MHz
*/

// Device configuration registers
_FGS(GWRP_OFF & GCP_OFF); // we don't want to write protect the flash
_FOSCSEL(FNOSC_FRC); // start up initially without the PLL but instead using the internal fast RC oscillator
_FOSC(FCKSM_CSECMD & OSCIOFNC_ON & POSCMD_EC);
_FWDT(FWDTEN_OFF); // not using the watchdog timer

signed short* local_record;
signed short* local_playback;
unsigned char local_record_pos, local_record_used;
unsigned char local_playback_pos, local_playback_used;

unsigned char stored_rates[MEM_WRAP_ADDR/(STORED_BYTES_PER_BUFFER/8)];
unsigned char local_playback_rates[LOCAL_BUFFERS];
unsigned char local_record_rates[LOCAL_BUFFERS];
unsigned char current_record_rate, next_record_rate;
extern unsigned char current_playback_rate;


static void setup_clock() {
  // set up clock for 40MHz
  PLLFBD = 38;
  CLKDIVbits.PLLPRE = 0;
#ifdef SLOW_CLOCK
  CLKDIVbits.PLLPOST = 1;
#else
  CLKDIVbits.PLLPOST = 0;
#endif
  __builtin_write_OSCCONH(0x03); // switch to PLL
  __builtin_write_OSCCONL(0x01);
}

static void setup_analog_input() {
  // RA0/AN1 = analog input
  AD1PCFGLbits.PCFG0 = 0;
  TRISAbits.TRISA0 = 1;
}

unsigned short mem_buf_record_pos;
unsigned short mem_buf_playback_pos;
#if defined(ADC_OVERSAMPLING) && ADC_OVERSAMPLING > 1
unsigned char mem_buf_partial_record_pos;
#endif

unsigned char ir_led, ir_led_timer;
unsigned short volume = 32768;
void ir_led_on() {
  ir_led = 1;
}
void ir_led_off() {
  ir_led = 0;
}

static unsigned char memory_ok() {
  unsigned short i;
  unsigned char j;
  
  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      sram_write_byte(j, 0x55 + i + j);
    } while( j );
  } while( i );

  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      if( sram_read_byte(j) != (unsigned char)(0x55 + i + j) )
        return 0;
    } while( j );
  } while( i );

  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      sram_write_byte(j, ~0x55 + i + j);
    } while( j );
  } while( i );

  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      if( sram_read_byte(j) != (unsigned char)(~0x55 + i + j) )
        return 0;
    } while( j );
  } while( i );

  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      sram_write_byte(j, 0);
    } while( j );
  } while( i );

  i = 0;
  do {
    sram_change_addr(i);
    sram_prepare_addr(++i);
    j = 8;
    do {
      --j;
      if( sram_read_byte(j) != 0 )
        return 0;
    } while( j );
  } while( i );

  return 1;
}

static unsigned short calculate_actual_delay() {
  unsigned short pos = mem_buf_playback_pos / (STORED_BYTES_PER_BUFFER/8);
  unsigned short finish = mem_buf_record_pos / (STORED_BYTES_PER_BUFFER/8);
  unsigned short ret = 0;
  while( pos != finish ) {
    switch(stored_rates[pos]) {
//  case 0:
    default:
      ret += 2;
      break;
    case 1:
      ret += 3;
      break;
    case 2:
      ret += 4;
      break;
    case 3:
      ret += 8;
      break;
    }
    if( ++pos == MEM_WRAP_ADDR / (STORED_BYTES_PER_BUFFER/8) )
      pos = 0;
  }
  return ret;
}

static unsigned char get_rate_from_actual_delay(unsigned short delay) {
  if( delay < 2636 )
    return 0;
  else if( delay < 3954 )
    return 1;
  else if( delay < 5272 )
    return 2;
  else
    return 3;
}

static unsigned short get_delay_pos_from_actual_delay(unsigned short delay, unsigned char rate) {
  switch(rate) {
//case 0:
  default:
    return delay/2;
  case 1:
    return delay/3;
  case 2:
    return delay/4;
  case 3:
    return delay/8;
  }
}

void set_record_rate(unsigned char rate) {
  adc_set_record_rate(rate);
  next_record_rate = rate;
}

int main(void) {
  unsigned char i, temp, flash_timer = 0, muted = 0;
  signed short* src, * dest;
  unsigned short cur_delay, eeprom_write_delay = 0, eeprom_volume_write_delay = 0;

  DataEEInit();
  cur_delay = DataEERead(0);
  if( cur_delay < 32 || cur_delay > 11648 )
    cur_delay = 32;
  else if( cur_delay >= 11520 )
    cur_delay = 11520;
  set_record_rate(get_rate_from_actual_delay(cur_delay));
  current_record_rate = next_record_rate;
  memset(stored_rates, current_record_rate, sizeof(stored_rates));
  memset(local_record_rates, current_record_rate, sizeof(local_record_rates));
  memset(local_playback_rates, current_record_rate, sizeof(local_playback_rates));
  cur_delay = get_delay_pos_from_actual_delay(cur_delay, current_record_rate);
  volume = DataEERead(1);
  if( volume > 32768 )
    volume = 32768;

  setup_clock();
  setup_analog_input();
  sram_init();
  init_ir();

  init_dac();
  init_adc();

  /*
    Interrupt priority:
      1. Change notification
      2. Timer 1
      3. DMA0
      4. DMA1
  */
  IPC4bits.CNIP = 7;
  IPC0bits.T1IP = 7;
  IPC1bits.DMA0IP = 6;
  IPC3bits.DMA1IP = 5;

  // set up Timer 2 for delay calculations
  T2CONbits.TCKPS = 1;//2;
  T2CONbits.TON = 1;

  mem_buf_record_pos = 0;
  mem_buf_playback_pos = MEM_WRAP_ADDR - (STORED_BYTES_PER_BUFFER/8) * cur_delay;//1800-45;
  local_record = (signed short*)malloc(SAMPLES_PER_BUFFER * sizeof(signed short) * LOCAL_BUFFERS);
  local_playback = (signed short*)malloc(SAMPLES_PER_BUFFER * sizeof(signed short) * LOCAL_BUFFERS);

  LATBbits.LATB0 = 1;
  LATBbits.LATB1 = 1;
  if( memory_ok() ) {
    LATBbits.LATB1 = 0;
  } else {
    LATBbits.LATB0 = 0;
    while(1)
      ;
  }

  while(1) {
    if( ir_final_code ) {
      switch( ir_final_code ) {
      case RC5(RC5_AR1729_VCR_VOLUP):
      case RC5(RC5_AR1729_VCR_VOLUP|IR_REPEAT):
		volume = (volume * 0x11111L)>>16;
        if( volume > 32768 )
          volume = 32768;
		eeprom_volume_write_delay = 160*10;
        break;
      case RC5(RC5_AR1729_VCR_VOLDN):
      case RC5(RC5_AR1729_VCR_VOLDN|IR_REPEAT):
		volume = (volume * 0xF000L)>>16;
        if( volume < 4096 )
          volume = 4096;
		eeprom_volume_write_delay = 160*10;
        break;
      case RC5(RC5_AR1729_VCR_CHDN):
      case RC5(RC5_AR1729_VCR_CHDN|IR_REPEAT):
        {
          unsigned long new_playback_pos = mem_buf_playback_pos + (STORED_BYTES_PER_BUFFER/8) * 8;
          unsigned long new_delay;
          if( new_playback_pos >= MEM_WRAP_ADDR )
            new_playback_pos -= MEM_WRAP_ADDR;
          new_delay = mem_buf_record_pos + MEM_WRAP_ADDR - new_playback_pos;
          while( new_delay >= MEM_WRAP_ADDR )
            new_delay -= MEM_WRAP_ADDR;
          if( new_delay > (STORED_BYTES_PER_BUFFER/8) * 32 ) {
            mem_buf_playback_pos = new_playback_pos;
			eeprom_write_delay = 160*10;
		  }
        }
		break;
      case RC5(RC5_AR1729_VCR_CHUP):
      case RC5(RC5_AR1729_VCR_CHUP|IR_REPEAT):
        {
          unsigned long new_playback_pos = mem_buf_playback_pos + MEM_WRAP_ADDR - (STORED_BYTES_PER_BUFFER/8) * 8;
          unsigned long new_delay;
          if( new_playback_pos >= MEM_WRAP_ADDR )
            new_playback_pos -= MEM_WRAP_ADDR;
          new_delay = mem_buf_record_pos + MEM_WRAP_ADDR - new_playback_pos;
          while( new_delay >= MEM_WRAP_ADDR )
            new_delay -= MEM_WRAP_ADDR;
          if( new_delay < MEM_WRAP_ADDR - (STORED_BYTES_PER_BUFFER/8) * 32 ) {
            mem_buf_playback_pos = new_playback_pos;
			eeprom_write_delay = 160*10;
          }
        }
		break;
      case RC5(RC5_AR1729_VCR_0):
      case RC5(RC5_AR1729_VCR_1):
      case RC5(RC5_AR1729_VCR_2):
      case RC5(RC5_AR1729_VCR_3):
      case RC5(RC5_AR1729_VCR_4):
      case RC5(RC5_AR1729_VCR_5):
      case RC5(RC5_AR1729_VCR_6):
      case RC5(RC5_AR1729_VCR_7):
      case RC5(RC5_AR1729_VCR_8):
      case RC5(RC5_AR1729_VCR_9):
        {
          unsigned char digit = ir_final_code - RC5(RC5_AR1729_VCR_0);
          unsigned long new_playback_pos = mem_buf_record_pos + MEM_WRAP_ADDR - ROUND_TO_MEMADDR( (SAMPLE_RATE*3/2) * (unsigned long)digit / 8 );
          unsigned long new_delay;
          if( ir_final_code == RC5(RC5_AR1729_VCR_0) )
            new_playback_pos = mem_buf_record_pos + MEM_WRAP_ADDR - (STORED_BYTES_PER_BUFFER/8) * 16;
          else if( ir_final_code == RC5(RC5_AR1729_VCR_9) )
            new_playback_pos = mem_buf_record_pos + (STORED_BYTES_PER_BUFFER/8) * 16;
          if( new_playback_pos >= MEM_WRAP_ADDR )
            new_playback_pos -= MEM_WRAP_ADDR;
          mem_buf_playback_pos = new_playback_pos;

          new_delay = mem_buf_record_pos + MEM_WRAP_ADDR - new_playback_pos;
          while( new_delay >= MEM_WRAP_ADDR )
            new_delay -= MEM_WRAP_ADDR;
          eeprom_write_delay = 160*10;
        }
        break;
      case RC5(RC5_AR1729_VCR_MUTE):
      case RC5(RC5_A1012_VCR_MUTE):
        muted = !muted;
        break;
      }
      ir_final_code = 0;
      ir_led_timer = 16;
    }
    if( ir_led ) {
      LATBbits.LATB1 = ir_led;
      if( ir_led && ir_led_timer < 8 )
        ir_led_timer = 8;
    } else if( ir_led_timer == 0 ) {
      LATBbits.LATB1 = muted ? ((flash_timer)>>4)&1 : 0;
    }

    asm volatile("disi #64");
    if( local_playback_used < LOCAL_BUFFERS ) {
      unsigned char pos = local_playback_pos;
      unsigned char write_head = pos + local_playback_used;
      unsigned short cur_volume = muted ? 0 : volume;
      ++local_playback_used;
      asm volatile("disi #0");
      if( write_head >= LOCAL_BUFFERS )
        write_head -= LOCAL_BUFFERS;

      i = SAMPLES_PER_BUFFER/16;
      dest = local_playback + write_head * SAMPLES_PER_BUFFER;
      local_playback_rates[pos] = stored_rates[mem_buf_playback_pos / (STORED_BYTES_PER_BUFFER/8)];
      do {
        sram_change_addr(mem_buf_playback_pos);
        sram_prepare_addr(++mem_buf_playback_pos);
        temp = sram_read_byte(0);
        *dest = __builtin_mulss(((signed short)sram_read_byte(1)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(2)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(3);
        *dest = __builtin_mulss(((signed short)sram_read_byte(4)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(5)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(6);
        *dest = __builtin_mulss(((signed short)sram_read_byte(7)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;

        sram_change_addr(mem_buf_playback_pos);
        sram_prepare_addr(++mem_buf_playback_pos);
        *dest = __builtin_mulss(((signed short)sram_read_byte(0)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(1);
        *dest = __builtin_mulss(((signed short)sram_read_byte(2)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(3)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(4);
        *dest = __builtin_mulss(((signed short)sram_read_byte(5)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(6)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(7);

        sram_change_addr(mem_buf_playback_pos);
        sram_prepare_addr(++mem_buf_playback_pos);
        *dest = __builtin_mulss(((signed short)sram_read_byte(0)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(1)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(2);
        *dest = __builtin_mulss(((signed short)sram_read_byte(3)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(4)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
        temp = sram_read_byte(5);
        *dest = __builtin_mulss(((signed short)sram_read_byte(6)<<4) | ((signed short)temp<<12), cur_volume)>>15;
		++dest;
        *dest = __builtin_mulss(((signed short)sram_read_byte(7)<<4) | ((signed short)(temp>>4)<<12), cur_volume)>>15;
		++dest;
      } while( --i );

      if( mem_buf_playback_pos == MEM_WRAP_ADDR )
        mem_buf_playback_pos = 0;

      continue;
    } else {
      asm volatile("disi #0");
    }

    asm volatile("disi #64");
    if( local_record_used > 0 && mem_buf_record_pos != mem_buf_playback_pos ) {
      unsigned char pos = local_record_pos;
      unsigned char read_tail = pos + LOCAL_BUFFERS - local_record_used;
      --local_record_used;
      asm volatile("disi #0");
      if( read_tail >= LOCAL_BUFFERS )
        read_tail -= LOCAL_BUFFERS;

      i = SAMPLES_PER_BUFFER/16;
      src = local_record + read_tail * SAMPLES_PER_BUFFER;
      stored_rates[mem_buf_record_pos / (STORED_BYTES_PER_BUFFER/8)] = local_record_rates[pos];
      do {
        sram_change_addr(mem_buf_record_pos);
        sram_prepare_addr(++mem_buf_record_pos);
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(0, temp);
		sram_write_byte(1, src[0]);
		sram_write_byte(2, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(3, temp);
		sram_write_byte(4, src[0]);
		sram_write_byte(5, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(6, temp);
		sram_write_byte(7, src[0]);

        sram_change_addr(mem_buf_record_pos);
        sram_prepare_addr(++mem_buf_record_pos);
		sram_write_byte(0, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(1, temp);
		sram_write_byte(2, src[0]);
		sram_write_byte(3, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(4, temp);
		sram_write_byte(5, src[0]);
		sram_write_byte(6, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(7, temp);

        sram_change_addr(mem_buf_record_pos);
        sram_prepare_addr(++mem_buf_record_pos);
		sram_write_byte(0, src[0]);
		sram_write_byte(1, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(2, temp);
		sram_write_byte(3, src[0]);
		sram_write_byte(4, src[1]);
		src += 2;
        temp = ((unsigned char)(src[0]>>8)&0x0f)|((unsigned char)(src[1]>>8)<<4);
		sram_write_byte(5, temp);
		sram_write_byte(6, src[0]);
		sram_write_byte(7, src[1]);
		src += 2;
      } while( --i );

      if( mem_buf_record_pos == MEM_WRAP_ADDR )
        mem_buf_record_pos = 0;

      continue;
    } else {
      asm volatile("disi #0");
    }

    if( local_playback_used == LOCAL_BUFFERS || local_record_used == 0 )
      pause_dac(0);
    else if( local_playback_used == 0 || local_record_used == LOCAL_BUFFERS )
      pause_dac(1);

    if( IFS0bits.T2IF ) {
      IFS0bits.T2IF = 0;

      if( ir_led_timer && !--ir_led_timer )
        ir_led = 0;
      if( eeprom_write_delay && !--eeprom_write_delay )
		DataEEWrite(calculate_actual_delay(), 0);
	  if( eeprom_volume_write_delay && !--eeprom_volume_write_delay )
		DataEEWrite(volume, 1);
      if( next_record_rate != current_playback_rate ) {
        LATBbits.LATB0 = ((flash_timer)>>4)&1;
      } else {
        LATBbits.LATB0 = 1;
      }
      ++flash_timer;
    }

    cur_delay = mem_buf_record_pos + MEM_WRAP_ADDR - mem_buf_playback_pos;
    while( cur_delay >= MEM_WRAP_ADDR )
      cur_delay -= MEM_WRAP_ADDR;
    if( next_record_rate == current_playback_rate && next_record_rate < 3 && cur_delay > (SAMPLE_RATE * 3 / 2 * 8 / 8) + 16 * SAMPLES_PER_BUFFER ) {
        set_record_rate( next_record_rate+1 );
        flash_timer = 0;
    } else if( next_record_rate == current_playback_rate && next_record_rate > 0 && cur_delay < (SAMPLE_RATE * 3 / 2 * 8 / 8) / 2 - 16 * SAMPLES_PER_BUFFER ) {
        set_record_rate( next_record_rate-1 );
        flash_timer = 0;
    }
  }

  return 0;
}
